/**
 * AegisRunner Selector Recorder - Content Script
 *
 * Injected into every page. When recording mode is active, it highlights hovered
 * elements and captures their optimal CSS selector on click.
 */

(() => {
  // Prevent double-injection
  if (window.__aegisRunnerSelectorRecorder) return;
  window.__aegisRunnerSelectorRecorder = true;

  const STORAGE_KEYS = {
    RECORDING: 'aegis_recording',
    SELECTORS: 'aegis_selectors',
  };

  const MAX_SELECTORS = 50;
  const HIGHLIGHT_CLASS = 'aegis-highlight';
  const TOAST_CLASS = 'aegis-toast';

  let recording = false;
  let hoveredElement = null;
  let toastTimeout = null;

  // --- Initialization ---

  init();

  async function init() {
    // Load initial recording state
    const data = await chrome.storage.local.get([STORAGE_KEYS.RECORDING]);
    recording = data[STORAGE_KEYS.RECORDING] || false;

    if (recording) {
      enableRecording();
    }

    // Listen for messages from popup
    chrome.runtime.onMessage.addListener(handleMessage);

    // Listen for storage changes (in case another tab toggles recording)
    chrome.storage.onChanged.addListener((changes, area) => {
      if (area === 'local' && changes[STORAGE_KEYS.RECORDING]) {
        const newVal = changes[STORAGE_KEYS.RECORDING].newValue;
        if (newVal && !recording) {
          recording = true;
          enableRecording();
        } else if (!newVal && recording) {
          recording = false;
          disableRecording();
        }
      }
    });
  }

  // --- Message Handling ---

  function handleMessage(message, sender, sendResponse) {
    if (message.type === 'AEGIS_TOGGLE_RECORDING') {
      recording = message.recording;
      if (recording) {
        enableRecording();
      } else {
        disableRecording();
      }
      sendResponse({ ok: true });
    }
    return true;
  }

  // --- Recording Mode ---

  function enableRecording() {
    document.addEventListener('mouseover', onMouseOver, true);
    document.addEventListener('mouseout', onMouseOut, true);
    document.addEventListener('click', onClick, true);
    document.body.style.cursor = 'crosshair';
  }

  function disableRecording() {
    document.removeEventListener('mouseover', onMouseOver, true);
    document.removeEventListener('mouseout', onMouseOut, true);
    document.removeEventListener('click', onClick, true);
    document.body.style.cursor = '';
    clearHighlight();
  }

  // --- Event Handlers ---

  function onMouseOver(e) {
    if (!recording) return;
    const el = e.target;

    // Ignore our own UI elements
    if (isAegisElement(el)) return;

    clearHighlight();
    hoveredElement = el;
    el.classList.add(HIGHLIGHT_CLASS);
  }

  function onMouseOut(e) {
    if (!recording) return;
    const el = e.target;
    if (el === hoveredElement) {
      el.classList.remove(HIGHLIGHT_CLASS);
      hoveredElement = null;
    }
  }

  function onClick(e) {
    if (!recording) return;

    const el = e.target;

    // Ignore our own UI elements
    if (isAegisElement(el)) return;

    // Prevent the default action and stop propagation so clicking
    // to select doesn't navigate, submit forms, etc.
    e.preventDefault();
    e.stopPropagation();
    e.stopImmediatePropagation();

    const selector = generateSelector(el);
    const tagName = el.tagName.toLowerCase();
    const textContent = (el.textContent || '').trim().substring(0, 60);

    // Copy to clipboard
    copyToClipboard(selector);

    // Store in history
    storeSelector({
      css: selector,
      tag: tagName,
      text: textContent,
      url: window.location.href,
      timestamp: Date.now(),
    });

    // Show toast
    showToast(`Selector copied: ${selector}`);

    // Clear highlight
    clearHighlight();

    return false;
  }

  // --- Selector Generation ---

  /**
   * Generate the optimal CSS selector for an element, following this priority:
   * 1. #id (if unique)
   * 2. [data-testid="..."]
   * 3. [aria-label="..."]
   * 4. tag.className combination (if unique)
   * 5. Fallback to nth-child path
   */
  function generateSelector(el) {
    // Skip html, body, head
    if (['html', 'body', 'head'].includes(el.tagName.toLowerCase())) {
      return el.tagName.toLowerCase();
    }

    // Strategy 1: Unique ID
    if (el.id) {
      const idSelector = `#${cssEscape(el.id)}`;
      if (isUnique(idSelector, el)) {
        return idSelector;
      }
    }

    // Strategy 2: data-testid
    const testId = el.getAttribute('data-testid');
    if (testId) {
      const selector = `[data-testid="${cssEscapeAttr(testId)}"]`;
      if (isUnique(selector, el)) {
        return selector;
      }
    }

    // Strategy 3: aria-label
    const ariaLabel = el.getAttribute('aria-label');
    if (ariaLabel) {
      const selector = `[aria-label="${cssEscapeAttr(ariaLabel)}"]`;
      if (isUnique(selector, el)) {
        return selector;
      }
    }

    // Strategy 4: tag + class combination
    const tagClassSelector = buildTagClassSelector(el);
    if (tagClassSelector && isUnique(tagClassSelector, el)) {
      return tagClassSelector;
    }

    // Strategy 5: Try tag + class within a parent context for shorter paths
    const contextSelector = buildContextSelector(el);
    if (contextSelector) {
      return contextSelector;
    }

    // Strategy 6: Full nth-child path
    return buildNthChildPath(el);
  }

  /**
   * Build a selector using tag name and class names.
   */
  function buildTagClassSelector(el) {
    const tag = el.tagName.toLowerCase();
    const classes = getFilteredClasses(el);

    if (classes.length === 0) {
      // Just tag name
      if (isUnique(tag, el)) return tag;
      return null;
    }

    // Try tag + all classes
    const fullSelector = tag + classes.map(c => `.${cssEscape(c)}`).join('');
    if (isUnique(fullSelector, el)) return fullSelector;

    // Try tag + individual classes
    for (const cls of classes) {
      const selector = `${tag}.${cssEscape(cls)}`;
      if (isUnique(selector, el)) return selector;
    }

    // Try just classes without tag
    for (const cls of classes) {
      const selector = `.${cssEscape(cls)}`;
      if (isUnique(selector, el)) return selector;
    }

    return fullSelector;
  }

  /**
   * Build a selector using nearest identifiable ancestor as context.
   * e.g., #sidebar > .nav-item > a
   */
  function buildContextSelector(el) {
    const parts = [];
    let current = el;
    let depth = 0;
    const maxDepth = 4;

    while (current && current !== document.documentElement && depth < maxDepth) {
      const part = getSelectorPart(current);
      parts.unshift(part);

      // Build candidate selector
      const candidate = parts.join(' > ');
      if (isUnique(candidate, el)) {
        return candidate;
      }

      // If this ancestor has an ID, we have enough context
      if (current.id) {
        break;
      }

      current = current.parentElement;
      depth++;
    }

    // Try the built path
    const candidate = parts.join(' > ');
    if (isUnique(candidate, el)) {
      return candidate;
    }

    return null;
  }

  /**
   * Get a simple selector part for a single element (for building paths).
   */
  function getSelectorPart(el) {
    const tag = el.tagName.toLowerCase();

    if (el.id) {
      return `#${cssEscape(el.id)}`;
    }

    const testId = el.getAttribute('data-testid');
    if (testId) {
      return `[data-testid="${cssEscapeAttr(testId)}"]`;
    }

    const classes = getFilteredClasses(el);
    if (classes.length > 0) {
      return tag + '.' + cssEscape(classes[0]);
    }

    return tag;
  }

  /**
   * Build a complete nth-child path from the element to a unique ancestor.
   */
  function buildNthChildPath(el) {
    const parts = [];
    let current = el;

    while (current && current !== document.documentElement && current !== document.body) {
      const tag = current.tagName.toLowerCase();
      const parent = current.parentElement;

      if (!parent) break;

      if (current.id) {
        parts.unshift(`#${cssEscape(current.id)}`);
        break;
      }

      const siblings = Array.from(parent.children).filter(
        c => c.tagName === current.tagName
      );

      if (siblings.length === 1) {
        parts.unshift(tag);
      } else {
        const index = siblings.indexOf(current) + 1;
        parts.unshift(`${tag}:nth-child(${getNthChildIndex(current)})`);
      }

      current = parent;
    }

    return parts.join(' > ') || el.tagName.toLowerCase();
  }

  /**
   * Get the nth-child index of an element within its parent.
   */
  function getNthChildIndex(el) {
    let index = 1;
    let sibling = el.previousElementSibling;
    while (sibling) {
      index++;
      sibling = sibling.previousElementSibling;
    }
    return index;
  }

  /**
   * Filter out classes that are ours or overly dynamic.
   */
  function getFilteredClasses(el) {
    return Array.from(el.classList).filter(cls => {
      // Filter out our own classes
      if (cls.startsWith('aegis-')) return false;
      // Filter out very long or hash-like classes (CSS modules, Tailwind JIT, etc.)
      if (cls.length > 40) return false;
      // Filter out classes that look like hashes (e.g., css-1a2b3c)
      if (/^[a-z]+-[a-z0-9]{5,}$/i.test(cls)) return false;
      return true;
    });
  }

  /**
   * Check if a selector uniquely matches the target element.
   */
  function isUnique(selector, target) {
    try {
      const matches = document.querySelectorAll(selector);
      return matches.length === 1 && matches[0] === target;
    } catch {
      return false;
    }
  }

  /**
   * Escape a string for use in CSS selectors.
   */
  function cssEscape(str) {
    if (typeof CSS !== 'undefined' && CSS.escape) {
      return CSS.escape(str);
    }
    // Fallback for environments without CSS.escape
    return str.replace(/([^\w-])/g, '\\$1');
  }

  /**
   * Escape a string for use in CSS attribute value selectors.
   */
  function cssEscapeAttr(str) {
    return str.replace(/\\/g, '\\\\').replace(/"/g, '\\"');
  }

  // --- Page ↔ Extension Bridge ---
  // Allow the AegisRunner web app to request captured selectors

  window.addEventListener('message', async (event) => {
    if (event.source !== window) return;

    if (event.data?.type === 'AEGIS_REQUEST_SELECTORS') {
      const data = await chrome.storage.local.get([STORAGE_KEYS.SELECTORS]);
      const selectors = data[STORAGE_KEYS.SELECTORS] || [];
      window.postMessage({
        type: 'AEGIS_SELECTORS_RESPONSE',
        selectors: selectors,
      }, '*');
    }

    if (event.data?.type === 'AEGIS_REMOVE_SELECTOR') {
      const { css, url } = event.data;
      const data = await chrome.storage.local.get([STORAGE_KEYS.SELECTORS]);
      let selectors = data[STORAGE_KEYS.SELECTORS] || [];
      selectors = selectors.filter(s => !(s.css === css && s.url === url));
      await chrome.storage.local.set({ [STORAGE_KEYS.SELECTORS]: selectors });
      window.postMessage({
        type: 'AEGIS_SELECTORS_RESPONSE',
        selectors: selectors,
      }, '*');
    }

    if (event.data?.type === 'AEGIS_PING') {
      window.postMessage({ type: 'AEGIS_PONG' }, '*');
    }
  });

  // Announce extension is available
  window.postMessage({ type: 'AEGIS_EXTENSION_READY' }, '*');

  // --- Helpers ---

  function isAegisElement(el) {
    if (!el || !el.closest) return false;
    return (
      el.classList.contains(TOAST_CLASS) ||
      el.closest(`.${TOAST_CLASS}`) !== null
    );
  }

  function clearHighlight() {
    if (hoveredElement) {
      hoveredElement.classList.remove(HIGHLIGHT_CLASS);
      hoveredElement = null;
    }
    // Also clean up any orphaned highlights
    document.querySelectorAll(`.${HIGHLIGHT_CLASS}`).forEach(el => {
      el.classList.remove(HIGHLIGHT_CLASS);
    });
  }

  // --- Clipboard ---

  function copyToClipboard(text) {
    try {
      navigator.clipboard.writeText(text);
    } catch {
      // Fallback using execCommand
      const textarea = document.createElement('textarea');
      textarea.value = text;
      textarea.style.position = 'fixed';
      textarea.style.left = '-9999px';
      textarea.style.top = '-9999px';
      document.body.appendChild(textarea);
      textarea.select();
      document.execCommand('copy');
      document.body.removeChild(textarea);
    }
  }

  // --- Storage ---

  async function storeSelector(entry) {
    const data = await chrome.storage.local.get([STORAGE_KEYS.SELECTORS]);
    let selectors = data[STORAGE_KEYS.SELECTORS] || [];

    // Check for duplicate
    const isDuplicate = selectors.some(
      s => s.css === entry.css && s.url === entry.url
    );
    if (isDuplicate) return;

    selectors.push(entry);

    // Cap at MAX_SELECTORS, removing oldest first
    if (selectors.length > MAX_SELECTORS) {
      selectors = selectors.slice(selectors.length - MAX_SELECTORS);
    }

    await chrome.storage.local.set({ [STORAGE_KEYS.SELECTORS]: selectors });
  }

  // --- Toast Notification ---

  function showToast(message) {
    // Remove existing toast
    const existing = document.querySelector(`.${TOAST_CLASS}`);
    if (existing) {
      existing.remove();
    }
    if (toastTimeout) {
      clearTimeout(toastTimeout);
    }

    const toast = document.createElement('div');
    toast.className = TOAST_CLASS;
    toast.textContent = message;
    document.body.appendChild(toast);

    // Trigger enter animation
    requestAnimationFrame(() => {
      toast.classList.add('aegis-toast-visible');
    });

    toastTimeout = setTimeout(() => {
      toast.classList.remove('aegis-toast-visible');
      toast.classList.add('aegis-toast-exit');
      setTimeout(() => {
        if (toast.parentNode) {
          toast.remove();
        }
      }, 300);
    }, 2500);
  }
})();
