/**
 * AegisRunner Selector Recorder - Popup Script
 *
 * Manages the popup UI: toggling recording mode, displaying recorded selectors,
 * copying selectors to clipboard, and sending selectors to AegisRunner.
 */

const STORAGE_KEYS = {
  RECORDING: 'aegis_recording',
  SELECTORS: 'aegis_selectors',
  AEGIS_URL: 'aegis_url',
};

const MAX_SELECTORS = 50;

let isRecording = false;
let selectors = [];

// --- Initialization ---

document.addEventListener('DOMContentLoaded', async () => {
  await loadState();
  renderUI();
  bindEvents();
});

async function loadState() {
  const data = await chrome.storage.local.get([
    STORAGE_KEYS.RECORDING,
    STORAGE_KEYS.SELECTORS,
    STORAGE_KEYS.AEGIS_URL,
  ]);
  isRecording = data[STORAGE_KEYS.RECORDING] || false;
  selectors = data[STORAGE_KEYS.SELECTORS] || [];

  const aegisUrlInput = document.getElementById('aegisUrl');
  if (data[STORAGE_KEYS.AEGIS_URL]) {
    aegisUrlInput.value = data[STORAGE_KEYS.AEGIS_URL];
  }
}

// --- Event Bindings ---

function bindEvents() {
  document.getElementById('toggleBtn').addEventListener('click', toggleRecording);
  document.getElementById('clearBtn').addEventListener('click', clearHistory);
  document.getElementById('aegisUrl').addEventListener('change', saveAegisUrl);

  // Listen for new selectors arriving from the content script
  chrome.storage.onChanged.addListener((changes, area) => {
    if (area === 'local' && changes[STORAGE_KEYS.SELECTORS]) {
      selectors = changes[STORAGE_KEYS.SELECTORS].newValue || [];
      renderSelectorList();
    }
  });
}

// --- Recording Toggle ---

async function toggleRecording() {
  isRecording = !isRecording;
  await chrome.storage.local.set({ [STORAGE_KEYS.RECORDING]: isRecording });

  // Notify the active tab's content script
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab?.id) {
      await chrome.tabs.sendMessage(tab.id, {
        type: 'AEGIS_TOGGLE_RECORDING',
        recording: isRecording,
      });
    }
  } catch (err) {
    // Content script may not be loaded on certain pages (chrome://, etc.)
    console.warn('Could not reach content script:', err.message);
  }

  renderUI();
}

// --- Clear History ---

async function clearHistory() {
  selectors = [];
  await chrome.storage.local.set({ [STORAGE_KEYS.SELECTORS]: [] });
  renderSelectorList();
}

// --- AegisRunner URL ---

async function saveAegisUrl(e) {
  const url = e.target.value.trim();
  await chrome.storage.local.set({ [STORAGE_KEYS.AEGIS_URL]: url });
}

// --- Selector Actions ---

async function copySelector(index) {
  const selector = selectors[index];
  if (!selector) return;

  try {
    await navigator.clipboard.writeText(selector.css);
    flashButton(index, 'copy', 'Copied!');
  } catch (err) {
    // Fallback: use a textarea
    const textarea = document.createElement('textarea');
    textarea.value = selector.css;
    document.body.appendChild(textarea);
    textarea.select();
    document.execCommand('copy');
    document.body.removeChild(textarea);
    flashButton(index, 'copy', 'Copied!');
  }
}

async function useInAegis(index) {
  const selector = selectors[index];
  if (!selector) return;

  const data = await chrome.storage.local.get([STORAGE_KEYS.AEGIS_URL]);
  let baseUrl = (data[STORAGE_KEYS.AEGIS_URL] || '').trim();

  if (!baseUrl) {
    baseUrl = 'https://aegisrunner.com';
  }

  // Remove trailing slash
  baseUrl = baseUrl.replace(/\/+$/, '');

  // Open the suites page with the selector pre-filled via query param
  const suiteUrl = `${baseUrl}/suites?selector=${encodeURIComponent(selector.css)}&element=${encodeURIComponent(selector.tag || '')}`;
  chrome.tabs.create({ url: suiteUrl });

  flashButton(index, 'use', 'Opened!');
}

async function removeSelector(index) {
  selectors.splice(index, 1);
  await chrome.storage.local.set({ [STORAGE_KEYS.SELECTORS]: selectors });
  renderSelectorList();
}

function flashButton(index, type, text) {
  const btn = document.querySelector(`[data-index="${index}"][data-action="${type}"]`);
  if (!btn) return;
  const original = btn.textContent;
  btn.textContent = text;
  setTimeout(() => {
    btn.textContent = original;
  }, 1200);
}

// --- Rendering ---

function renderUI() {
  renderToggleButton();
  renderStatusBadge();
  renderSelectorList();
}

function renderToggleButton() {
  const btn = document.getElementById('toggleBtn');
  const text = document.getElementById('toggleText');

  if (isRecording) {
    btn.classList.add('active');
    text.textContent = 'Stop Recording';
  } else {
    btn.classList.remove('active');
    text.textContent = 'Start Recording';
  }
}

function renderStatusBadge() {
  const badge = document.getElementById('statusBadge');
  const statusText = document.getElementById('statusText');

  if (isRecording) {
    badge.className = 'status-badge recording';
    statusText.textContent = 'Recording';
  } else {
    badge.className = 'status-badge paused';
    statusText.textContent = 'Paused';
  }
}

function renderSelectorList() {
  const listEl = document.getElementById('selectorList');
  const countEl = document.getElementById('selectorCount');
  const emptyState = document.getElementById('emptyState');

  countEl.textContent = `${selectors.length} item${selectors.length !== 1 ? 's' : ''}`;

  if (selectors.length === 0) {
    listEl.innerHTML = '';
    listEl.appendChild(createEmptyState());
    return;
  }

  listEl.innerHTML = '';

  // Show most recent first
  const reversed = [...selectors].reverse();
  reversed.forEach((sel, reverseIndex) => {
    const actualIndex = selectors.length - 1 - reverseIndex;
    const item = createSelectorItem(sel, actualIndex);
    listEl.appendChild(item);
  });
}

function createEmptyState() {
  const div = document.createElement('div');
  div.className = 'empty-state';
  div.id = 'emptyState';
  div.innerHTML = `
    <svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
      <path d="M15 15l-2 5L9 9l11 4-5 2zm-6.364-.364a9 9 0 1112.728 0"></path>
    </svg>
    <p>No selectors recorded yet.<br>Click "Start Recording" and click elements on any page.</p>
  `;
  return div;
}

function createSelectorItem(sel, index) {
  const item = document.createElement('div');
  item.className = 'selector-item';

  const selectorText = document.createElement('div');
  selectorText.className = 'selector-text';
  selectorText.textContent = sel.css;

  const meta = document.createElement('div');
  meta.className = 'selector-meta';

  const tag = document.createElement('span');
  tag.className = 'selector-tag';
  tag.textContent = sel.tag || 'element';

  const actions = document.createElement('div');
  actions.className = 'selector-actions';

  const copyBtn = document.createElement('button');
  copyBtn.className = 'btn-small btn-copy';
  copyBtn.textContent = 'Copy';
  copyBtn.setAttribute('data-index', index);
  copyBtn.setAttribute('data-action', 'copy');
  copyBtn.addEventListener('click', () => copySelector(index));

  const useBtn = document.createElement('button');
  useBtn.className = 'btn-small btn-use';
  useBtn.textContent = 'Use in AegisRunner';
  useBtn.setAttribute('data-index', index);
  useBtn.setAttribute('data-action', 'use');
  useBtn.addEventListener('click', () => useInAegis(index));

  const removeBtn = document.createElement('button');
  removeBtn.className = 'btn-small btn-remove';
  removeBtn.textContent = 'Remove';
  removeBtn.setAttribute('data-index', index);
  removeBtn.setAttribute('data-action', 'remove');
  removeBtn.addEventListener('click', () => removeSelector(index));

  actions.appendChild(copyBtn);
  actions.appendChild(useBtn);
  actions.appendChild(removeBtn);

  meta.appendChild(tag);
  meta.appendChild(actions);

  item.appendChild(selectorText);
  item.appendChild(meta);

  return item;
}
